// File created: 2007-10-09 19:55:17

package ope.adventure;

import java.io.FileInputStream;
import java.io.InputStream;
import java.io.IOException;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import ope.adventure.book.Library;
import ope.adventure.parsers.BookParser;
import ope.adventure.parsers.SpellParser;
import ope.adventure.parsers.SummonSpellParser;
import ope.adventure.util.Utils;
import ope.adventure.util.VFS;

// Handles loading data from files and the like. Also stores that data.
public final class Configuration {
	private static final String CONFIG_PATH = "config.cfg";
	private static final String DEFAULT_CFG = "D.";
	private VFS fileManager;

	private       Library            library;
	private final Map<String, Spell> spells = new HashMap<String, Spell>();
	private       StringBuilder      assignment;
	private       int                entityCount, assignmentCount;

	public Library            getLibrary()    { return library; }
	public Map<String, Spell> getSpells()     { return spells; }
	public String             getAssignment() { return assignment.toString(); }
	public boolean            hasBooks()      { return library.size() != 0; }
	public boolean            hasSpells()     { return !spells.isEmpty(); }

	public static String getHelp() {
		return
			"The configuration file is called " + CONFIG_PATH + " and should " +
			"be found in the current\ndirectory. If it is not found, the string " +
			"\"" + DEFAULT_CFG + "\" is used as the configuration\nfile." +
			"\n\n" +
			"The file consists simply of a sequence of lines, whose meanings " +
			"depend on the\nfirst character on the line as follows:\n" +
			"\tD - the rest of the line is considered a path to a directory,\n" +
			"\tF - the rest of the line is considered a path to a ZIP file.\n" +
			"\n" +
			"The directory's or file's contents are searched for data - that " +
			"is, books,\nspells, entities, and assignments. All are collected " +
			"together and used as\nin-game data." +
			"\n\n" +
			"Books are in books/ and have the extension .book.txt.\n" +
			"Spells are in spells/ and have the extension .spell.txt.\n" +
			"Entities are in entities/ and have the extension .entity.txt.\n" +
			"Assignments are in assignments/ and have .assignment.txt.\n" +
			"\n" +
			"Note that there mustn't be any file name conflicts, due to " +
			"internal\nlimitations: each file name must appear only once.";
	}

	public Configuration() {
		reconfigure();
	}

	public void reconfigure() {
		String conf = DEFAULT_CFG;
		try {
			final InputStream file = new FileInputStream(CONFIG_PATH);
			conf = Utils.getContents(file);
			file.close();
		} catch (final IOException e) {
			conf = "D.";
		}

		fileManager = new VFS();

		for (final String line : Utils.lineSplit(conf))
		switch (line.charAt(0)) {
			case 'D':
				fileManager.addDir(line.substring(1, line.length()));
				break;
			case 'F':
				fileManager.addFile(line.substring(1, line.length()));
				break;
			default: break;
		}

		loadData();

		System.err.printf("Configuration :: %d books%n", library.size());
		System.err.printf("Configuration :: %d spells%n", spells.size());
		System.err.printf("Configuration :: %d entities%n", entityCount);
		System.err.printf("Configuration :: %d assignments%n", assignmentCount);
	}

	private void loadData() {
		library = new Library();
		spells.clear();
		entityCount = assignmentCount = 0;
		assignment = new StringBuilder(
			"Sum-1.3581: Advanced Summoning in Practice\n" +
			"FINAL PROJECT ASSIGNMENT\n" +
			"========================\n" +
			"\n" +
			"Using all you've learned so far with the pre-drawn sigils in the " +
			"laboratory,\nsummon an entity from another plane. When it is " +
			"contained, ring the bell in the\nlobby and an assistant will " +
			"grade you." +
			"\n\n" +
			"You will be graded on the power of the creature and how well it " +
			"is held in\nplace. Something you may have conjured back in high " +
			"school will not earn you\nmany points, nor will an entity which is " +
			"rampaging beyond your control." +
			"\n\n"
		);

		try {
			loadBooks();
			loadSpells();
			loadEntities();
			loadAssignment();
		} catch (final Exception e) {
			System.err.printf(
				"Configuration :: %s while attempting to load data%n", e);
		}

		assignment = new StringBuilder(assignment.toString().trim());
		assignment.trimToSize();
	}

	// FIXME: refactor these methods
	private void loadBooks() {
		final String path = "books/";
		final Set<String> added = new HashSet<String>();

		for (final String bookName : fileManager.requestDirContents(path)) {
			if (!bookName.endsWith(".book.txt"))
				continue;

			if (added.contains(bookName)) {
				System.err.printf(
					"Configuration :: attempted reload of book '%s'%n",
					bookName);
				continue;
			}

			try {
				library.addBook(
					BookParser.parse(
						fileManager.requestFileContents(path + bookName)));

				added.add(bookName);
			}
			// as long as we get at least one book we're good, just notify
			// about any not found
			catch (final Exception e) {
				System.err.printf(
					"Configuration :: %s while reading book '%s'%n",
					e, bookName);
			}
		}
	}
	private void loadSpells() {
		final String path = "spells/";
		final Set<String> added = new HashSet<String>();

		for (final String spellName : fileManager.requestDirContents(path)) {
			if (!spellName.endsWith(".spell.txt"))
				continue;

			if (added.contains(spellName)) {
				System.err.printf(
					"Configuration :: attempted reload of spell '%s'%n",
					spellName);
				continue;
			}

			try {
				final Spell s = SpellParser.parse(
					fileManager.requestFileContents(path + spellName));

				spells.put(s.getName(), s);

				added.add(spellName);

			} catch (final Exception e) {
				System.err.printf(
					"Configuration :: %s while reading spell '%s'%n",
					e, spellName);
			}
		}
	}
	private void loadEntities() {
		final String path = "entities/";
		final Set<String> added = new HashSet<String>();

		for (final String entityName : fileManager.requestDirContents(path)) {
			if (!entityName.endsWith(".entity.txt"))
				continue;

			if (added.contains(entityName)) {
				System.err.printf(
					"Configuration :: attempted reload of entity '%s'%n",
					entityName);
				continue;
			}

			try {
				final Spell s = SummonSpellParser.parse(
					fileManager.requestFileContents(path + entityName));

				spells.put(s.getName(), s);

				added.add(entityName);
				++entityCount;

			} catch (final Exception e) {
				System.err.printf(
					"Configuration :: %s while reading entity '%s'%n",
					e, entityName);
			}
		}
	}
	private void loadAssignment() {
		final String path = "assignments/";
		final Set<String> added = new HashSet<String>();

		for (final String fn : fileManager.requestDirContents(path)) {
			if (!fn.endsWith(".assignment.txt"))
				continue;

			if (added.contains(fn)) {
				System.err.printf(
					"Configuration :: attempted reload of assignment '%s'%n",
					fn);
				continue;
			}

			try {
				assignment.append(fileManager.requestFileContents(path + fn));

				added.add(fn);
				++assignmentCount;

			} catch (final Exception e) {
				System.err.printf(
					"Configuration :: %s while reading assignment '%s'%n",
					e, fn);
			}
		}
	}
}
