// File created: 2007-10-25 16:19:16

package ope.adventure;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Random;
import java.util.Set;

import ope.adventure.util.BooleanExpression;

public final class Entity {
	private static final Random random = new Random();

	private final Set<String> names = new HashSet<String>();
	private       int         enteredDefaultStateTime, summonTime;

	private final List<State> states = new ArrayList<State>();
	private       State       defaultState;
	private       State       currentState;

	private boolean leaving = false;

	public void summon(final int t) {
		summonTime = t;
		endMotion(t);
	}

	public void addState(final State s) {
		if (states.isEmpty())
			defaultState = s;

		states.add(s);
	}

	public void addName(final String n) { names.add(n); }
	public Set<String> getNames() { return names; }

	public String getDescription() { return currentState.getDescription(); }
	public String getAppearance()  { return currentState.getAppearance(); }

	public boolean isLeaving() { return leaving; }

	// the endgame score
	public byte getWorth() {
		byte b = defaultState.getWorth();

		if (currentState != defaultState)
			b += currentState.getWorth();

		return b;
	}

	// response to a spell
	public String responseTo(
		final Set<String> effs,
		final Player p,
		final int time
	) {
		if (currentState != defaultState) {
			final String stateResponse = currentState.responseTo(effs, p, time);
			if (stateResponse != null)
				return stateResponse;
		}

		final String defaultResponse = defaultState.responseTo(effs, p, time);
		if (defaultResponse != null)
			return defaultResponse;

		return "";
	}

	public Effect beginMotion(
		final int prev, final int now,
		final Set<String> effects // effects in the room the Entity is in
	) {
		// one motion at a time
		if (currentState != defaultState)
			return null;

		for (final State state : states) {
			if (state == defaultState)
				continue;

			if (!state.getPrerequisites().matches(effects))
				continue;

			// checks = number of values in the half-open interval (prev, now]
			//          which are multiples of state.getCheckFrequency(), when
			//          counting from enteredDefaultStateTime.
			// The proof is left as an exercise for the reader.
			int checks =
				  (now  - enteredDefaultStateTime) / state.getCheckFrequency()
				- (prev - enteredDefaultStateTime) / state.getCheckFrequency();

			// override, we want to check as soon as we are summoned
			if (now == summonTime)
				checks = 1;

			// FIXME
			// This isn't fully correct: it now checks each state in order,
			// when it should check them in the order of when the check frequency
			// is arrived at.
			// I.e. if 20 seconds pass, and we have two states with frequencies of
			// 6 and 5, the first state is checked three times before the second
			// is checked at all. What should happen is that the second state is
			// checked once, the first once, the second once, the first once, the
			// second once, the first once, and the second for the fourth time.

			// Oh well, it'll have to do for now.

			while (checks-- > 0)
			if (random.nextDouble() <= state.getProbability()) {

				currentState = state;
				return new Effect(
					null, // no name
					null, // no description: the Entity has it
					// HACK: pass entry message as expiration message, it's ignored
					// for motions and provides a handy way of printing it only once
					// when the motion is begun
					state.getEntryMessage(),
					now + state.getLength(),
					state.getFinalResults());
			}
		}

		return null;
	}

	public void endMotion(final int time) {
		currentState = defaultState;
		enteredDefaultStateTime = time;
	}

	public final class State {
		// entryMessage is the message printed when entering this state
		// for the default state, it is null.
		private final String description, appearance, entryMessage;

		private final int interval, length;
		private final double probability;

		private final byte worth;

		// processResults are used whenever a response is needed while doing the
		// motion, finalResults are used when the motion is complete
		private final List<Result> processResults, finalResults;

		private final BooleanExpression prerequisites;

		// for the default state only
		public State(
			final String d, final String a,
			final byte w,
			final List<Result> r
		) {
			description = d;
			appearance = a;
			worth = w;
			processResults = r;

			entryMessage = null;
			interval = length = 0;
			probability = Double.NaN;
			finalResults = null;
			prerequisites = null;
		}

		public State(
			final String d, final String a, final String e,
			final int i, final double p, final int t,
			final byte w,
			final List<Result> pr, final List<Result> fr,
			final BooleanExpression pre
		) {
			description = d;
			appearance = a;
			entryMessage = e;
			interval = i;
			probability = p;
			length = t;
			worth = w;
			processResults = pr;
			finalResults = fr;
			prerequisites = pre;
		}

		public String            getDescription()    { return description; }
		public String            getAppearance()     { return appearance; }
		public String            getEntryMessage()   { return entryMessage; }
		public int               getLength()         { return length; }
		public int               getCheckFrequency() { return interval; }
		public double            getProbability()    { return probability; }
		public byte              getWorth()          { return worth; }
		public List<Result>      getFinalResults()   { return finalResults; }
		public BooleanExpression getPrerequisites()  { return prerequisites; }

		public String responseTo(
			final Set<String> effects,
			final Player player,
			final int time
		) {
			final Result res = Result.getResult(processResults, effects);

			if (res == null)
				return "";

			if (!res.isSuccessful()) {
				Entity.this.endMotion(time);
				player.getRoom().removeMotions();
			}

			final Result.Consequences consequences = res.getConsequences();

			consequences.apply(player, null);

			if (consequences.willLeave())
				leaving = true;

			return res.getMessage();
		}
	}
}
