// File created: 2007-10-23 21:47:16

package ope.adventure;

import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import ope.adventure.util.BooleanExpression;

// What happens when a spell/summoning/motion is completed.
public final class Result {
	// Each individual happening, a Result usually contains more than one.
	public static final class Consequence {
		public enum Type {
			COUNTER,      // spell arg is countered
			SANITY,       // arg sanity gained
			GAMEOVER,     // happens only if player is in same room
			REALGAMEOVER, // always happens
			LEAVE,        // motion only: entity leaves
			SUMMON,       // spell only: summons arg

			// not consequences per se
			ACTIVE_MSG,
			EXPIRE_MSG
		}

		private final Type   type;
		private final Object arg;

		public Consequence(final Type t) {
			this(t, null);
		}
		public Consequence(final Type t, final Object o) {
			type = t;
			arg = o;
		}

		private Type   getType() { return type; }
		private Object getArg()  { return arg; }
	}

	public static final class Consequences {
		private final List<String> counters = new ArrayList<String>();
		private String  description, expiration;
		private Entity  entity;
		private byte    sanity;
		private boolean quit, alwaysQuit, leave;

		public List<String> getCounters()          { return counters; }
		public String       getDescription()       { return description; }
		public String       getExpirationMessage() { return expiration; }
		public Entity       getEntity()            { return entity; }
		public byte         getSanity()            { return sanity; }
		public boolean      willQuit()             { return quit || alwaysQuit; }
		public boolean      willAlwaysQuit()       { return alwaysQuit; }
		public boolean      willLeave()            { return leave; }

		public void apply(final Player player, final Integer time) {
			apply(player, time, player.getRoom());
		}
		public void apply(
			final Player player,
			final Integer time, // not int, so that it can be null
			final Room room
		) {
			if (alwaysQuit || (quit && player.getRoom() == room)) {
				// early return: if game ending, doesn't matter what else happens
				player.endGame();
				return;
			}

			if (entity != null) {
				// should be checked in Spell
				assert room.summoningAllowed();
				assert room.getEntity() == null;

				entity.summon(time);
				room.addEntity(entity);
			}

			for (final String counter : counters)
				room.counterSpell(counter);

// SAN			player.modifySanity(sanity);
		}
	}

	// Results are parsed incrementally, so these can't be final and set in a
	// constructor... hence also the set* methods, which are only meant to be
	// called once (though it's not enforced)
	private BooleanExpression condition;
	private String            message;
	private boolean           success = true;

	private final List<Consequence>
		consequenceList = new ArrayList<Consequence>();

	private Consequences consequences;

	public BooleanExpression getCondition() { return condition; }
	public String            getMessage()   { return message; }
	public boolean           isSuccessful() { return success; }

	public void setCondition(final BooleanExpression bexp) {
		condition = bexp;
	}
	public void setMessage(final String s) {
		message = s;
	}
	public void setSuccess(final boolean b) {
		success = b;
	}

	public void addConsequence(final Consequence.Type type) {
		consequenceList.add(new Consequence(type));
	}
	public void addConsequence(final Consequence.Type type, final Object o) {
		consequenceList.add(new Consequence(type, o));
	}
	public void removeConsequence(final Consequence c) {
		consequenceList.remove(c);
	}

	// never null: does a lazy init
	public Consequences getConsequences() {
		if (consequences != null)
			return consequences;

		consequences = new Consequences();

		for (final Consequence c : consequenceList)
		switch (c.getType()) {
			case ACTIVE_MSG:
				consequences.description = (String)c.getArg();
				break;

			case EXPIRE_MSG:
				consequences.expiration = (String)c.getArg();
				break;

			case COUNTER:
				consequences.counters.add((String)c.getArg());
				break;

			case SUMMON:
				consequences.entity = (Entity)c.getArg();
				break;

			case SANITY:
				consequences.sanity = (Byte)c.getArg();
				break;

			case GAMEOVER:
				consequences.quit = true;
				break;

			case REALGAMEOVER:
				consequences.alwaysQuit = true;
				break;

			case LEAVE:
				consequences.leave = true;
				break;

			default:
				throw new IllegalStateException(
					"Result :: invalid consequence, " +
					"should have been caught by parser");
		}
		return consequences;
	}

	// finds the first matching Result in the given List
	public static Result getResult(
		final List<Result> possibilities,
		final Set<String> effects
	) {
		for (final Result res : possibilities)
			if (res.getCondition().matches(effects))
				return res;
		return null;
	}
}
