// File created: 2007-10-18 14:14:27

package ope.adventure;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.PriorityQueue;
import java.util.Queue;
import java.util.Set;

import ope.adventure.util.Utils;

public final class Room extends Area {
	private final Map<Room, Integer> distances = new HashMap<Room, Integer>();
	private final String[]           descs;
	private final boolean            gotLibrary;
	private final boolean            canSummon;
	private       Entity             entity;

	private       Queue<Effect> effectQueue = new PriorityQueue<Effect>();
	private final Map<String, Integer>
		effectMap = new HashMap<String, Integer>();

	public Room(
		final String n, final String[] ds,
		final boolean lib, final boolean sum
	) {
		super.setName(n);
		descs = ds;
		gotLibrary = lib;
		canSummon = sum;
	}

	public boolean hasLibrary()       { return gotLibrary; }
	public boolean summoningAllowed() { return canSummon; }

   public Entity getEntity()               { return entity; }
   public void   addEntity(final Entity e) { entity = e; }
   public void removeEntity() {
   	entity = null;
		removeMotions();
	}

	public int distanceTo(final Room r) { return distances.get(r); }

	public void addEffect(final Effect e) {
		effectQueue.add(e);

		final String name = e.getName();
		if (name != null) {
			final Integer instances = effectMap.get(name);

			if (instances == null)
				effectMap.put(name, 1);
			else
				effectMap.put(name, instances + 1);
		}
	}

	public boolean hasExpiringEffects(final int time) {
		if (effectQueue.isEmpty())
			return false;

		// lazily remove countered spells here
		// countered spells are removed from effectMap, but not effectQueue
		// motions have a null name, so don't remove those
		Effect e;
		for (;;) {
			e = effectQueue.peek();
			if (
				e.getEndingTime() > time ||
				e.getName() == null      ||
				effectMap.containsKey(e.getName())
			)
				break;
			effectQueue.remove();
		}

		return e.getEndingTime() <= time;
	}

	public Effect pollEffect() {
		final Effect e = effectQueue.poll();

		final String name = e.getName();
		if (name != null) {
			final int i = effectMap.get(name) - 1;
			assert i >= 0;
			if (i == 0)
				effectMap.remove(name);
			else
				effectMap.put(name, i);
		}

		return e;
	}

	public void counterSpell(final String spell) { effectMap.remove(spell); }

	public void removeMotions() {
   	// easy to do due to an implementation detail:
   	// they're (only) in effectQueue, and their name is null
   	if (!effectQueue.isEmpty()) {
			final Queue<Effect>
				newQueue = new PriorityQueue<Effect>(effectQueue.size());

			for (final Effect e : effectQueue)
				if (e.getName() != null)
					newQueue.add(e);

			effectQueue = newQueue;
		}
	}

	public Set<String> getEffectSet() { return effectMap.keySet(); }

	public static List<Room> getRooms()              { return ROOMS; }
	public static Room       getRoom(final String n) { return ROOM_MAP.get(n); }
	public static Room       getStartingRoom()       { return START; }

// SAN   // getFullDescription() takes no arguments, UI-mandated
// SAN	// hence, we use a currentDescIdx, modified according to sanity
// SAN	public static void setSanityLevel(final byte san) {
// SAN
// SAN		// formula works for every case except sanity = 0, when it gives
// SAN		// currentDescIdx = descs.length, which of course doesn't work.
// SAN		// however, at sanity < 1 the player dies so it doesn't matter...
// SAN		assert san > 0;
// SAN
// SAN		currentDescIdx = descCount * (Player.MAX_SANITY - san)/Player.MAX_SANITY;
// SAN	}

	public String getFullDescription() {
		final StringBuilder desc = new StringBuilder(descs[currentDescIdx]);

		for (final Effect e : effectQueue) {
			final String s = e.getDescription();
			if (s != null)
				desc.append("\n\n").append(s);
		}

		if (entity != null)
			desc.append("\n\n").append(entity.getDescription());

		return desc.toString();
	}

	private static       int currentDescIdx = 0;
// SAN	private static final int descCount;

	private static final Map<String, Room> ROOM_MAP;
	private static final List<Room>        ROOMS;

	private static final Room START;

	static {
		final String[]
			LOBBY_NAMES = {"Lobby",      "Lob"},
			LIB_NAMES   = {"Library",    "Lib"},
			LAB_NAMES   = {"Laboratory", "Lab"};

		final String[]
			LOBBY_DESC = {
				"A nondescript and unimportant room serving only as the entry " +
				"to the library and\nthe laboratory." +
				"\n\n" +
				"An archway leads to the library and stairs go down to the " +
				"laboratory. There is\nalso a door leading out, but you're not " +
				"going that way." +
				"\n\n" +
				"The assignment is stapled to the wall.\n" +
				"The notification bell levitates near the door."
			},
			LIB_DESC = {
				"Shelves upon shelves of books of all shapes and sizes adorn " +
				"this room, which\nseems to never end. You've received enough " +
				"library training to know that all\nlibraries are interconnected " +
				"in such a way that you can find every book ever\nwritten in any " +
				"library, but you never understood how that can possibly work." +
				"\n\n" +
				"Fortunately you do know how to work it, at least to an extent: " +
				"just because the\nlibrary's contents are infinite in number " +
				"doesn't mean you couldn't find and\nread what you wanted." +
				"\n\n" +
				"An archway leads to the lobby and stairs go down to the " +
				"laboratory."
			},
			LAB_DESC = {
				"The place for doing some serious magic. Your eyes are drawn, as " +
				"always, to the\nfloor: enchanted so as to always be in pristine " +
				"condition, and marked with so\nmany crisscrossing sigils that " +
				"it's practically impossible to identify any one.\nIt looks " +
				"almost like random doodlings, but you've studied enough to know " +
				"that\nevery one is there for a reason. So nice of the " +
				"university to provide all the\nknown sigils: it's a pain to " +
				"draw them by yourself." +
				"\n\n" +
				"Two sets of stairs lead up, one to the lobby and the other to " +
				"the library."
			};

// SAN		descCount = LIB_DESC.length;

// SAN		assert descCount == LOBBY_DESC.length;
// SAN		assert descCount == LAB_DESC.length;

		final Room lobby = new Room(LOBBY_NAMES[0], LOBBY_DESC, false, false);
		final Room lib   = new Room(LIB_NAMES[0],   LIB_DESC,    true, false);
		final Room lab   = new Room(LAB_NAMES[0],   LAB_DESC,   false,  true);

		ROOM_MAP = new HashMap<String, Room>(3, 1f);

		for (final String n : LOBBY_NAMES)
			ROOM_MAP.put(Utils.toLower(n), lobby);
		for (final String n : LIB_NAMES)
			ROOM_MAP.put(Utils.toLower(n),   lib);
		for (final String n : LAB_NAMES)
			ROOM_MAP.put(Utils.toLower(n),   lab);

		ROOMS = new ArrayList<Room>(3);

		ROOMS.add(lobby);
		ROOMS.add(lib);
		ROOMS.add(lab);

		setDistanceBetween(lobby, lib, 5);
		setDistanceBetween(lobby, lab, 5);
		setDistanceBetween(lib,   lab, 8);

// SAN		setSanityLevel(Player.MAX_SANITY);

		START = lobby;
	}

	private static void setDistanceBetween(
		final Room r1, final Room r2,
		final int d
	) {
		r1.distances.put(r2, d);
		r2.distances.put(r1, d);
	}
}
