// File created: 2007-10-19 15:29:47

package ope.adventure.actions;

import java.util.ArrayList;
import java.util.List;

import ope.adventure.Action;
import ope.adventure.Configuration;
import ope.adventure.Player;
import ope.adventure.book.Book;
import ope.adventure.book.BookSection;
import ope.adventure.book.ID;
import ope.adventure.util.Utils;

// "browse" [book-identifier]
// "browse" book-identifier ["for"] keyword
public final class BrowseAction extends Action {
	public boolean doableWithoutBooks () { return false; }
	public boolean doableWithoutSpells() { return true; }
	public boolean stopReading        () { return false; }

	public String getShortHelp() {
		return "list keywords of book or sections containing keyword";
	}
	public String getLongHelp() {
		return
			"\"browse\" book-identifier\n" +
			"\"browse\" book-identifier [\"for\"] keyword\n" +
			"\n" +
			"If given only a book identifier, list the keywords within the " +
			"book. Note that\nnot all books have indices." +
			"\n\n" +
			"If given a keyword, list the sections within the book which " +
			"contain the given\nkeyword. Note that this is is the only way of " +
			"accessing the contents of a book:\nnobody wants to read these tomes" +
			"through from beginning to end!" +
			"\n\n" +
			"The output format is \"<section-identifier>. <snippet>\". Section " +
			"identifiers\ncan be passed to the \"read\" verb." +
			"\n\n" +
			"Overwrites \"recent\".";
	}

	public int perform(
		final List<String> words,
		final Configuration conf, final Player player
	) {
		if (!player.getRoom().hasLibrary()) {
			println("There is no library here to browse.");
			return 0;
		}

		Book book;

		if (words.isEmpty()) {
			if (player.isReading())
				book = player.topBook();
			else {
				println(
					"You browse your mind for some useful information " +
					"and come up with nothing.");
				return 1;
			}
		} else {
			final String bookId = words.get(0);
			book = ID.getBook(bookId, player.getPrevBooks());

			if (book == null) {
				println("Invalid book identifier. Check 'recent' for valid ones.");
				return 1;
			}
			if (!player.isReading(book)) {
				println("That book is still on the shelf.");
				return 0;
			}
		}

		if (words.size() <= 1)
			return listKeywords(book);
		else {
			int idx = 1;
			if (words.size() > 2 && words.get(1).equals("for"))
				idx = 2;

			return listSectionsContaining(
				Utils.toLower(Utils.join(words.subList(idx, words.size()), " ")),
				book, player);
		}
	}

	private int listKeywords(final Book book) {
		if (book.hasIndex()) {
			printfln(
				"The index of '%s' lists the following words:",
				book.getShortTitle());

			for (final String kw : book.getKeywords())
				printfln("        %s", kw);
		} else
			printfln("'%s' doesn't appear to have an index.",
				book.getShortTitle());

		return 1;
	}

	private int listSectionsContaining(
		final String keyword,
		final Book book, final Player player
	) {
		final List<BookSection> sections = book.getSections(keyword);
		player.setNextSections(sections);

		if (sections == null)
			printfln(
				"'%s' doesn't contain the keyword '%s'.",
				book.getShortTitle(), keyword);
		else {
			printfln(
				"The following sections in '%s' contain the keyword '%s':",
				book.getShortTitle(), keyword);

			List<String> nextSnippets = player.getPrevSnippets();
			if (nextSnippets == null)
				nextSnippets = new ArrayList<String>();
			else
				nextSnippets.clear();

			ID.resetSectionId();
			for (final BookSection sec : sections) {
				final String snippet = sec.getSnippet(keyword);
				nextSnippets.add(snippet);
				printSectionId(snippet);
			}

			player.setNextSnippets(nextSnippets);
		}
		return 2;
	}
}
