// File created: 2007-10-19 15:51:10

package ope.adventure.actions;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import ope.adventure.Action;
import ope.adventure.Configuration;
import ope.adventure.Player;
import ope.adventure.book.Book;
import ope.adventure.book.ID;
import ope.adventure.book.Library;
import ope.adventure.util.Utils;

// "search" ["for"] "title" book-name
// "search" ["for"] keyword
public final class SearchAction extends Action {
	public boolean doableWithoutBooks () { return false; }
	public boolean doableWithoutSpells() { return true; }
	public boolean stopReading        () { return false; }

	public String getShortHelp() {
		return "list books with matching title or keyword";
	}
	public String getLongHelp() {
		return
			"\"search\" [\"for\"] \"title\" book-name\n" +
			"\"find\" [\"for\"] \"title\" book-name\n" +
			"\"search\" [\"for\"] keyword\n" +
			"\"find\" [\"for\"] keyword\n" +
			"\n" +
			"If using \"title\" form, output a list of books whose titles " +
			"contain book-name." +
			"\n\n" +
			"Otherwise, output a list of books which contain the given keyword." +
			"\n\n" +
			"The output format is \"<book-identifier>. <title>\" in both cases. " +
			"Book\nidentifiers can be passed to the \"read\" verb." +
			"\n\n" +
			"Overwrites \"recent\".";
	}

	public int perform(
		final List<String> words,
		final Configuration conf, final Player player
	) {
		if (!player.getRoom().hasLibrary()) {
			println("There is no library here to search.");
			return 0;
		}
		if (words.isEmpty()) {
			println(
				"You scour your thoughts for that thing you've been thinking of, "+
				"but fail to find it.");
			return 1;
		}

		int idx = 0;

		if (words.get(0).equals("for"))
			++idx;

		final Library library = conf.getLibrary();

		if (words.size() > idx && words.get(idx).equals("title")) {
			++idx;

			return findBooksWithMatchingTitle(
				player, library,
				Utils.toLower(Utils.join(words.subList(idx, words.size()), " ")));
		} else
			return findBooksWithMatchingKeyword(
				player, library,
				Utils.toLower(Utils.join(words.subList(idx, words.size()), " ")));
	}

	private int findBooksWithMatchingTitle(
		final Player player, final Library library, final String word
	) {
		printfln(
			"You come up with the following books whose titles contain '%s':",
			word);

		List<Book> nextBooks = player.getPrevBooks();
		if (nextBooks == null)
			nextBooks = new ArrayList<Book>();
		else
			nextBooks.clear();

		ID.resetBookId();
		for (final Book book : library)
		if (Utils.toLower(book.getName()).contains(word)) {
			nextBooks.add(book);
			printBookId(book);
		}

		if (nextBooks.isEmpty())
			println("You found no such books.");

		player.setNextBooks(nextBooks);

		return library.size();
	}

	private int findBooksWithMatchingKeyword(
		final Player player, final Library library, final String word
	) {
		printfln(
			"You find the following books containing the keyword '%s':",
			word);

		final List<Book> books = library.getByKeyword(word);
		player.setNextBooks(books);

		if (books == null)
			println("You found no such books.");
		else {
			Collections.sort(books);
			ID.resetBookId();
			for (final Book book : books)
				printBookId(book);
		}

		return library.size() * 3 / 2;
	}
}
