// File created: 2007-10-23 12:14:38

package ope.adventure.parsers;

import java.util.ArrayList;
import java.util.InputMismatchException;
import java.util.List;
import java.util.Scanner;
import java.util.regex.Pattern;

import ope.adventure.Result;
import ope.adventure.Spell;
import ope.adventure.util.BooleanExpression;
import ope.adventure.util.Utils;

public final class SpellParser {
	private SpellParser() {}

	private static String name;
	private static int castingTime, duration;

	private static final Pattern
		VALID_NAME = Pattern.compile(
			"^[A-Za-z\\-']+(?: [A-Za-z\\-']+)*$");

	public static Spell parse(final String textFile) {
		final List<Result> results = new ArrayList<Result>();

		final Scanner scan = new Scanner(textFile);

		name = readName(scan);

		scan.useDelimiter("\\s+");

		castingTime = scan.nextInt();
		duration    = scan.nextInt();

		scan.useDelimiter("$");

		final String rest = scan.next();
		int i = 0;
		boolean gotDefault = false;

		do	try {
			final Result res = ResultParser.parse(
				rest.substring(i, rest.length()),
				SpellConsequencePredicate.getInstance());

			i += ResultParser.getEnd();

			if (res.getCondition().getSym() == BooleanExpression.Symbol.DEFAULT)
				gotDefault = true;

			results.add(res);

		} catch (final Exception e) {
			System.err.printf("SpellParser :: %s while parsing %s%n", e, name);

			while (i < rest.length() && rest.charAt(i) != '}')
				++i;
			++i;

		} while (i < rest.length());

		if (!gotDefault)
			throw new BadSpellException("spell must have default result");

		return new Spell(name, castingTime, duration, results);
	}

	static String readName(final Scanner scan) {
		final String name = Utils.readLineSkipComments(scan);

		if (!VALID_NAME.matcher(name).matches())
			throw new BadSpellException("spell name must contain only [A-Za-z ]");

		return name;
	}
}

final class BadSpellException extends InputMismatchException {
	public BadSpellException(final String msg) {
		super(msg);
	}
}

final class SpellConsequencePredicate
	extends ResultParser.ConsequencePredicate {

	// singleton
	private static SpellConsequencePredicate
		instance = new SpellConsequencePredicate();

	public static SpellConsequencePredicate getInstance() {
		return instance;
	}

	public boolean accept(final String cons, final Result res) {
		return
			acceptFailure(cons, res) ||
			acceptStr("counter", Result.Consequence.Type.COUNTER,    cons, res) ||
			acceptStr("active",  Result.Consequence.Type.ACTIVE_MSG, cons, res) ||
			acceptStr("expire",  Result.Consequence.Type.EXPIRE_MSG, cons, res) ||
			acceptByte("sanity", Result.Consequence.Type.SANITY,     cons, res) ||
			acceptJust("gameover", Result.Consequence.Type.GAMEOVER, cons, res) ||
			acceptJust(
				"realgameover", Result.Consequence.Type.REALGAMEOVER, cons, res);
	}
}
