// File created: 2007-10-26 15:32:51

package ope.adventure.parsers;

import java.util.ArrayList;
import java.util.InputMismatchException;
import java.util.List;

import ope.adventure.Entity;
import ope.adventure.Result;
import ope.adventure.util.BooleanExpression;
import ope.adventure.util.Utils;

public final class StateParser {
	private StateParser() {}

	private static int pos;

	public static int getEnd() { return pos; }

	private static int freq, length;
	private static double prob;
	private static byte worth;
	private static String desc, app, entry;
	private static List<Result> whileResults, endResults;
	private static BooleanExpression prerequisites;

	public static Entity.State parseDefaultState(
		final String str,
		final Entity ent
	) {
		init();

		worth = readByte(str);

		desc = readString(str);
		app  = readString(str);

		// read results until hit numbers/EOF
		for (
			pos = Utils.skipWhiteSpace(pos, str);

			pos < str.length() &&
			!(str.charAt(pos) <= '9' && str.charAt(pos) >= '0');

			pos = Utils.skipWhiteSpace(pos, str)
		) {
			final Result res = ResultParser.parse(
				str.substring(pos, str.length()),
				StateConsequencePredicate.getInstance());

			pos += ResultParser.getEnd();

			if (res.getCondition().getSym() == BooleanExpression.Symbol.DEFAULT)
				throw new BadStateException(
					"default states cannot have default results");

			whileResults.add(res);
		}

		return ent.new State(desc, app, worth, whileResults);
	}

	public static Entity.State parse(final String str, final Entity ent) {
		init();

		freq   = readInt(str);
		prob   = readDbl(str);
		length = readInt(str);

		if (freq <= 0)
			throw new BadStateException("frequency must be greater than zero");

		// not problematic like frequency, just stupid
		if (prob <= 0)
			System.err.println("StateParser :: warning, probability <= 0");
		if (prob > 1)
			System.err.println("StateParser :: warning, probability > 1");

		final BooleanExpressionParser
			bexp = new BooleanExpressionParser(str.substring(pos, str.length()));
		prerequisites = bexp.parse();
		pos += bexp.getEnd();

		worth = readByte(str);

		entry = readString(str);
		desc  = readString(str);
		app   = readString(str);

		boolean endGotDefault = false;

		// read 'while' and 'end' results until hit numbers/EOF
		for (
			pos = Utils.skipWhiteSpace(pos, str);

			pos < str.length() &&
			!(str.charAt(pos) <= '9' && str.charAt(pos) >= '0');

			pos = Utils.skipWhiteSpace(pos, str)
		) {
			boolean whileResult;

			if (str.substring(pos, str.length()).startsWith("while")) {
				whileResult = true;
				pos += "while".length();

			} else if (str.substring(pos, str.length()).startsWith("end")) {
				whileResult = false;
				pos += "end".length();
			} else
				throw new BadStateException(
					"non-default states's results must be preceded by " +
					"'while' or 'end'");

			final Result r = ResultParser.parse(
				str.substring(pos, str.length()),
				StateConsequencePredicate.getInstance());

			pos += ResultParser.getEnd();

			if (whileResult) {
				if (r.getCondition().getSym() == BooleanExpression.Symbol.DEFAULT)
					throw new BadStateException(
						"'while' results cannot have default as a condition");

				whileResults.add(r);
			} else {
				if (r.getCondition().getSym() == BooleanExpression.Symbol.DEFAULT)
					endGotDefault = true;

				endResults.add(r);
			}
		}

		if (!endGotDefault)
			throw new BadStateException("end of state must have default result");

		return ent.new State(
			desc, app, entry,
			freq, prob, length,
			worth,
			whileResults, endResults,
			prerequisites);
	}

	private static void init() {
		whileResults  = new ArrayList<Result>();
		endResults    = new ArrayList<Result>();
		pos = 0;
	}

	// " string "
	private static String readString(final String str) {
		pos = Utils.skipWhiteSpace(pos, str);

		if (str.charAt(pos++) != '"')
			throw new BadStateException("expected '\"' to begin string");

		final int start = pos;

		while (str.charAt(pos) != '"')
			++pos;

		return str.substring(start, pos++);
	}

	// must be positive
	private static int readInt(final String str) {
		pos = Utils.skipWhiteSpace(pos, str);

		final int start = pos;

		for (;;) {
			final char c = str.charAt(pos);
			if (c > '9' || c < '0')
				break;
			++pos;
		}

		return Integer.parseInt(str.substring(start, pos));
	}

	// must be positive
	private static double readDbl(final String str) {
		pos = Utils.skipWhiteSpace(pos, str);

		final int start = pos;

		boolean gotDot = false;
		char c;
		do {
			c = str.charAt(pos++);
			if (!gotDot && c == '.') {
				gotDot = true;
				c = str.charAt(pos++);
			}
		} while (c <= '9' && c >= '0');

		return Double.parseDouble(str.substring(start, pos));
	}

	// more often negative than not
	private static byte readByte(final String str) {
		pos = Utils.skipWhiteSpace(pos, str);

		final int start = pos;

		if (str.charAt(pos) == '-')
			++pos;

		for (;;) {
			final char c = str.charAt(pos);
			if (c > '9' || c < '0')
				break;
			++pos;
		}

		return Byte.parseByte(str.substring(start, pos));
	}
}

final class BadStateException extends InputMismatchException {
	BadStateException(final String msg) {
		super(msg);
	}
}

final class StateConsequencePredicate
	extends ResultParser.ConsequencePredicate {

	// singleton
	private static StateConsequencePredicate
		instance = new StateConsequencePredicate();

	public static StateConsequencePredicate getInstance() {
		return instance;
	}

	public boolean accept(final String cons, final Result res) {
		return
			acceptFailure(cons, res) ||
			acceptStr("counter",   Result.Consequence.Type.COUNTER,  cons, res) ||
			acceptByte("sanity",   Result.Consequence.Type.SANITY,   cons, res) ||
			acceptJust("leave",    Result.Consequence.Type.LEAVE,    cons, res) ||
			acceptJust("gameover", Result.Consequence.Type.GAMEOVER, cons, res) ||
			acceptJust(
				"realgameover", Result.Consequence.Type.REALGAMEOVER, cons, res);
	}
}
