// File created: 2007-10-04 16:43:50

package ope.adventure.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.InputStream;
import java.io.IOException;
import java.util.Arrays;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashSet;
import java.util.List;
import java.util.NoSuchElementException;
import java.util.Set;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;

/** A simple virtual file system. Basic usage: add directory and ZIP file names
 * with addDir and addFile then request files by passing a name to the
 * requestFileContents method. The first readable file found in the directories
 * is opened and its contents returned. Failing that, the ZIP files are
 * searched.
 *
 * Directory contents may be requested with requestDirContents. Note that this
 * is much slower: requestFileContents simply returns the first match, whereas
 * requestDirContents always goes through every directory of the specified name.
 *
 * One may add any number of paths and ZIP files to be searched.
 *
 * All paths are searched before ZIP files, and both paths and ZIP files are
 * searched in the order in which they are added.
 *
 * Not thread safe.
 */
public final class VFS {
	private final Set<CharSequence> dirNames = new HashSet<CharSequence>();
	private final Set<CharSequence> zipFiles = new HashSet<CharSequence>();

	public void addDir (final CharSequence path) { dirNames.add(path); }
	public void addFile(final CharSequence file) { zipFiles.add(file); }

	public int size() { return dirNames.size() + zipFiles.size(); }

	public String requestFileContents(final CharSequence fileName)
		throws IOException {

		for (final CharSequence path : dirNames) {

			final StringBuilder fullPath = new StringBuilder(path);
			fullPath.append(File.separatorChar).append(fileName);

			final File wanted = new File(fullPath.toString());

			if (wanted.isFile() && wanted.canRead()) {
				final InputStream contents = new FileInputStream(wanted);
				final String retval = Utils.getContents(contents);
				contents.close();
				return retval;
			}
		}

		for (final CharSequence file : zipFiles) {
			final ZipFile zipFile = new ZipFile(file.toString());
			final ZipEntry wanted = zipFile.getEntry(fileName.toString());

			if (wanted != null)
				return Utils.getContents(zipFile.getInputStream(wanted));
		}

		throw new NoSuchElementException(
			"VFS :: file '" + fileName + "' not found."
		);
	}

	// Duplicates are possible and are returned in a nondeterministic order.
	// Doing a requestFileContents on every String returned may well result
	// in getting the same file multiple times.
	public String[] requestDirContents(final CharSequence dirPath) {

		final List<String> contents = new ArrayList<String>();

		for (final CharSequence path : dirNames) {

			final StringBuilder fullPath = new StringBuilder(path);
			fullPath.append(File.separatorChar).append(dirPath);

			final File dir = new File(fullPath.toString());

			if (dir.isDirectory() && dir.canRead())
				contents.addAll(Arrays.asList(dir.list()));
		}

		for (final CharSequence file : zipFiles)
		try {
			final ZipFile zipFile = new ZipFile(file.toString());

			final Enumeration<? extends ZipEntry> entries = zipFile.entries();

			while (entries.hasMoreElements()) {
				final ZipEntry entry = entries.nextElement();
				if (
					entry.getName().startsWith(dirPath.toString()) &&
					!entry.isDirectory()
				) {
					 contents.add(new File(entry.getName()).getName());
				 }
			}
		} catch (final IOException e) {
			System.err.printf("VFS :: %s while requesting %s%n", e, dirPath);
		}

		return contents.toArray(new String[contents.size()]);
	}
}
